package control;

import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import abstraction.Action;
import abstraction.OnlyReadIterator;
import abstraction.Service;
import abstraction.State;

/**
 * @author Balestra Concetta, Qusa Hani
 * @version 1.2 09/04/05
 * @since JDK 1.6
 */

/**
 * This Class contains algorithms useful for simulation
 */
public class Simulation {
		
	private Set<SimulatedBy> pairs;
	private Service target;
	private Community community;
	
	protected Simulation(Service target, Community community) {
		this.target=target;
		this.community=community;
		pairs = new HashSet<SimulatedBy>();
	}
	
	/**
	 * creates the final simulation set containing all pairs of states which are in simulation
	 */
	protected void compositionViaSimulation () {
		this.createSimulationSet();
		boolean modified = true;
		while (!pairs.isEmpty() && modified) {
			modified = checkSimulationSet();
		}
	}
	
	/**
	 * creates an initial simulation set containing:
	 *  - all pairs of final target state and final community state
	 *  - all pairs of not final target state and community state
	 */
	private void createSimulationSet (){
		Iterator<State> ts = this.target.getStates();
		while (ts.hasNext()) {
			State t = (State)ts.next();
			if(this.target.isFinalState(t)) {
				Iterator<CState> fcs = this.community.getFinalCommunityStates();
				while(fcs.hasNext()) {
					CState s = (CState)fcs.next(); 
					SimulatedBy newRecord = new SimulatedBy(t, s);
					this.pairs.add(newRecord);	
				}
			}
			else {
				Iterator<CState> cs = this.community.getCommunityStates();
				while(cs.hasNext()) {
					CState s1 = (CState)cs.next(); 
					SimulatedBy newRecord = new SimulatedBy(t, s1);
					this.pairs.add(newRecord);	
				}
			}
		}
		
	}		
	
	/**
	 * controls if all records in simulation set are in simulation and if one of them isn't it is deleted
	 * @return boolean, true if there is a record that isn't in simulation and that has been removed from the simulation set; false otherwise
	 */
	private boolean checkSimulationSet() {
		
		Iterator<SimulatedBy> record = pairs.iterator();
		while (record.hasNext()) {
			SimulatedBy line = record.next();
			State ts = line.getTargetState();
		    CState cs = line.getCommunityState();
		    Iterator<Action> action = this.target.getActions(ts);
	        while(action.hasNext()) {
	        	Action as = action.next();
	        	Iterator<State> nextState = this.target.getNextStates(ts, as);// contains only one next state, because target is deterministic
	        	while (nextState.hasNext()) {
	        		State nextTState = (State)nextState.next();	
		        	if (!checkSimulationOfNextState(cs, as, nextTState)) {
		        		this.pairs.remove (line);
		        		return true;
		        	}
		       	}
	        }
		}
		return false;
	}
	
	/**
	 * checks if there is a state of the community going with the selected action to another state 
	 * that is in simulation with the next state of the target
	 * @param cs - list of states which represent a community state
	 * @param as - action
	 * @param nextTState - next target state
	 * @return boolean, true if there is a next state of the community in simulation with the next state of the target; false otherwise
	 */
	private boolean checkSimulationOfNextState (CState cs, Action as, State nextTState) {
		for (int i = 0; i < cs.getSize(); i++){
			Service serv = this.community.getService(i);
			
			if (serv.containsPresentAction(cs.get(i), as)) {
				boolean candoit = true;
				Iterator<State> nextState = serv.getNextStates(cs.get(i), as);
				while (nextState.hasNext() && candoit) {
					State next = (State)nextState.next();
					CState hs = cs.buildNewState (i, next );
					SimulatedBy rec = new SimulatedBy (nextTState, hs);
					if (!this.contains(rec)) {
						candoit = false;
					}
				}
				if(candoit) return true;
			}
		}
		return false;
	}
	


	/**
	 * checks if the given pair of states is present in the simulation set
	 * @param sb - element to check 
	 * @return boolean true if the set contains sb, false otherwise
	 */
	protected boolean contains (SimulatedBy sb) {
		return this.pairs.contains(sb);
	}
	
	/**
	 * gets an iterator on the element of the simulation set
	 * @return Iterator on SimulatedBy element
	 */
	protected Iterator<SimulatedBy> getSimulationSet () {
		return new OnlyReadIterator<SimulatedBy>(this.pairs.iterator());
	}
	
	protected boolean isEmpty() {
		return this.pairs.isEmpty();
	}
	
	public String toString () {
		StringBuilder s = new StringBuilder();
		Iterator<SimulatedBy> it = pairs.iterator();
		while (it.hasNext()) {
			s = s.append("(" + ((SimulatedBy)it.next()).toString()+ ")"); 
		}
		return s.toString();
	}
}


